<?php

namespace Iranserver\Models\Traits;

use Illuminate\Database\Eloquent\Model;
use ReflectionClass;
use ValueError;

/**
 * Do you want to upgrade to PHP ^8.1?
 * I totally understand the excitement! However, if you're still unable to upgrade, don't worry!
 * In PHP <= 8.0, you can simulate enums using a handy trait.
 * Simply define a simple class and use simple constants as enum cases.
 * By doing so, you can enjoy all the benefits and behaviors of enums in your code.
 * So, keep coding and unleashing the power of PHP!
 */
trait EnumSimulator
{
    public $name;
    public $value;

    public static function cases(): array
    {
        $object = new ReflectionClass(static::class);
        $return = [];
        foreach ($object->getConstants() as $name => $value) {
            $return[] = tap(new static, function ($enum) use ($name, $value) {
                $enum->name = $name;
                $enum->value = $value;
            });
        }
        return $return;
    }

    public static function from($value)
    {
        foreach (static::cases() as $case) if ($case->value == $value) return $case;
        throw new ValueError(sprintf('"%s" is not a valid backing value for enum "%s"', $value, static::class));
    }

    public static function tryFrom($value)
    {
        foreach (static::cases() as $case) if ($case->value == $value) return $case;
        return null;
    }

    public function get(Model $model, string $key, $value, array $attributes)
    {
        if (is_null($value)) {
            return null;
        }

        if ($value instanceof static) {
            return $value;
        }

        return static::from($value);
    }

    public function set(Model $model, string $key, $value, array $attributes)
    {
        if ($value instanceof static) {
            return $value->value;
        }

        return static::from($value)->value;
    }
}