<?php

namespace Iranserver\Metrics;

use Illuminate\Console\Events\ScheduledTaskFailed;
use Illuminate\Console\Events\ScheduledTaskStarting;
use Illuminate\Queue\Events\JobProcessing;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Laravel\Lumen\Routing\Router;
use Prometheus\CollectorRegistry;
use Prometheus\Storage\Redis;
use Illuminate\Console\Events\ScheduledTaskFinished;
use Illuminate\Console\Scheduling\Event;

class MetricsServiceProvider extends ServiceProvider
{
    private static $scheduleStartTime = [];

    public function register()
    {
        $this->mergeConfigFrom(__DIR__ . "/config/metrics.php", 'metrics');
        $this->registerMetricService();
        $this->registerMetricRoutes();
        $this->registerMetricCommands();
        $this->registerListeners();
    }

    public function registerMetricService()
    {
        $this->app->singleton(CollectorRegistry::class, function () {
            $default = config('metrics.default');
            $config = config("metrics.drivers.$default");
            Redis::setPrefix(Arr::pull($config, 'prefix'));
            return new CollectorRegistry(new Redis($config));
        });
    }

    public function registerMetricRoutes()
    {
        $router = $this->app['router'];

        if ($router instanceof Router) { // Lumen router
            $router->get("/metrics", 'Iranserver\Metrics\LumenMetricsController@metrics');
        }
        else { // Laravel router
            Route::get("/metrics", [LaravelMetricsController::class, 'metrics']);
        }
    }

    public function registerMetricCommands()
    {
        $this->commands(MakeMetricCommand::class);
    }

    public function registerListeners()
    {
        // CommandStarting::class
        // JobProcessing::class
        // ScheduledTaskStarting::class

        $this->app['events']->listen(JobProcessing::class, function (JobProcessing $event) {
            $jobs = Metrics::getOrRegisterGauge('application', 'job_last_run_time', 'time of jobs process in laravel', ['name']);
            $jobs->set(microtime(true), [
                get_class($event->job)
            ]);
        });

        $this->app['events']->listen(ScheduledTaskStarting::class, function (ScheduledTaskStarting $command) {
            $jobs = Metrics::getOrRegisterGauge('application', 'scheduled_last_run_time', 'time of scheduled process in laravel', ['name']);
            $jobs->set(self::$scheduleStartTime[$command->task->command] = microtime(true), [
                $command->task->command
            ]);
        });

        $this->app['events']->listen(ScheduledTaskFinished::class, [$this, 'scheduledTaskCompleteHandler']);
        $this->app['events']->listen(ScheduledTaskFailed::class, [$this, 'scheduledTaskCompleteHandler']);
    }

    public function scheduledTaskCompleteHandler($command)
    {
        $jobs = Metrics::getOrRegisterGauge('application', 'scheduled_last_run_status', 'last status of scheduled process in laravel', ['name']);
        $jobs->set($command instanceof ScheduledTaskFinished ? 0 : 1, [
            $command->task->command
        ]);

        if (isset(self::$scheduleStartTime[$command->task->command])) {
            $jobs = Metrics::getOrRegisterGauge('application', 'scheduled_duration', 'duration of scheduled process in laravel', ['name']);
            $jobs->set(round(microtime(true) - self::$scheduleStartTime[$command->task->command], 3), [
                $command->task->command
            ]);
        }
    }
}
