<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Tests\Feature;

use Illuminate\Support\Facades\Bus;
use Jaeger\Thrift\Agent\Zipkin\BinaryAnnotation;
use OpenTracing\Formats;
use OpenTracing\Tracer;
use Iranserver\LaravelJaeger\Middleware\JobTracingMiddleware;
use Iranserver\LaravelJaeger\Services\Job\JobInjectionMaker;
use Iranserver\LaravelJaeger\Services\Job\JobWithTracingInjectionDispatcher;
use Iranserver\LaravelJaeger\Services\Span\SpanCreator;
use Iranserver\LaravelJaeger\Tests\JaravelTestCase;
use Iranserver\LaravelJaeger\Tests\Utils\TestJob;

class JobTracingTest extends JaravelTestCase
{
    public function testJobHandledWithInjection()
    {
        $tracingContextField = JobTracingMiddleware::JOB_TRACING_CONTEXT_FIELD;
        $fakeBus = Bus::fake();
        $bus = new JobWithTracingInjectionDispatcher($fakeBus, $this->app->make(JobInjectionMaker::class));

        $spanCreator = app(SpanCreator::class);
        $tracer = app(Tracer::class);

        $span = $spanCreator->create('Call MyService');

        $traceId = $span->getContext()->getTraceId();

        $bus->dispatch(new TestJob());

        $fakeBus->assertDispatched(TestJob::class,
            function ($job) use ($tracer, $traceId) {
                return $tracer->extract(Formats\TEXT_MAP, $job->{$tracingContextField})->getTraceId() === (int)$traceId;
            });
    }

    public function testJobMiddlewareWithoutContext()
    {
        $job = new TestJob();

        $middleware = $this->app->make(JobTracingMiddleware::class);

        $middleware->handle($job, function () {
            return true;
        });

        $spans = $this->reporter->getSpans();

        $this->assertCount(1, $spans);
        $span = $spans[0];

        $this->assertSame('Job: Iranserver\LaravelJaeger\Tests\Utils\TestJob', $span->getOperationName());

        $tags = collect($span->getTags())->mapWithKeys(function (BinaryAnnotation $tag) {
            return [$tag->key => $tag->value];
        });

        $expectedTags = [
            'type' => 'job',
            'job_class' => 'Iranserver\LaravelJaeger\Tests\Utils\TestJob',
        ];

        $this->assertSame($expectedTags, $tags->intersect($expectedTags)->toArray());
    }

    public function testJobMiddlewareWithContext()
    {
        $injectionMaker = $this->app->make(JobInjectionMaker::class);

        $job = new TestJob();

        $tracer = $this->app->make(Tracer::class);
        $spanCreator = $this->app->make(SpanCreator::class);

        $middleware = $this->app->make(JobTracingMiddleware::class);

        $spanCreator->create('Call MyService');
        $job = $injectionMaker->injectParentSpanToCommand($job);

        $middleware->handle($job, function() {
            return true;
        });

        optional($tracer->getScopeManager()->getActive())
            ->close();
        $tracer->flush();

        $spans = array_reverse($this->reporter->getSpans());

        $this->assertCount(2, $spans);

        $serviceSpan = $spans[0];
        $jobSpan = $spans[1];

        $this->assertSame('Call MyService', $serviceSpan->getOperationName());
        $this->assertSame('Job: Iranserver\LaravelJaeger\Tests\Utils\TestJob', $jobSpan->getOperationName());

        $this->assertSame(
            $serviceSpan->getContext()->getSpanId(), $jobSpan->getContext()->getParentId()
        );
    }

}
