<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Listeners\Job;

use Illuminate\Queue\Events\JobFailed;
use Illuminate\Queue\Events\JobProcessed;
use Iranserver\LaravelJaeger\Services\Exceptions\CallerException;
use Iranserver\LaravelJaeger\Services\Span\ParentSpans;
use Iranserver\LaravelJaeger\Services\Span\SpanTagHelper;
use OpenTracing\Tracer;
use Iranserver\LaravelJaeger\Services\Caller;
use Illuminate\Support\Facades\Config as ConfigRepository;

final class JobFinishedListener
{
    private $tracer;

    public function __construct(Tracer $tracer)
    {
        $this->tracer = $tracer;
    }

    /**
     * @param JobProcessed|JobFailed $event
     * @throws CallerException
     */
    public function handle($event)
    {
        $jobSpan = ParentSpans::get();
        if (!$jobSpan) {
            return;
        }
        $spanName = Caller::call(ConfigRepository::get('jaeger.job.span_name'), [$event]);
        if ($jobSpan->getOperationName() !== $spanName) {
            return;
        }

        SpanTagHelper::setTags(
            $jobSpan,
            Caller::call(ConfigRepository::get('jaeger.job.tags'), [$event])
        );

        if ($event instanceof JobFailed) {
            $jobSpan->setTag('error', true);
            $jobSpan->log([
                'exception.message' => $event->exception->getMessage(),
                'exception.code' => $event->exception->getCode(),
                'exception.line' => $event->exception->getLine(),
                'exception.file' => $event->exception->getFile(),
            ]);
        }

        $jobSpan->finish();
        $this->tracer->flush();

        ParentSpans::remove($jobSpan);
    }
}
