<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Services\Job;

use Iranserver\LaravelJaeger\Services\Caller;
use Iranserver\LaravelJaeger\Services\Exceptions\CallerException;
use Iranserver\LaravelJaeger\Services\Span\SpanCreator;
use Iranserver\LaravelJaeger\Services\Span\SpanTagHelper;
use OpenTracing\Formats;
use OpenTracing\Tracer;
use Illuminate\Support\Facades\Config;
use OpenTracing\Reference;

class JobInjectionMaker
{
    private $tracer;
    private $spanCreator;

    public function __construct(Tracer $tracer, SpanCreator $spanCreator)
    {
        $this->tracer = $tracer;
        $this->spanCreator = $spanCreator;
    }

    /**
     * @throws CallerException
     */
    public function injectParentSpanToCommand(object $command, string $methodName): object
    {
        if (!empty($command->jaegerTracingContext)) {
            return $command;
        }

        $parentSpan = $this->tracer->getActiveSpan();

        if (!$parentSpan) {
            return $command;
        }

        $span = $this->spanCreator->create(
            Caller::call(Config::get('jaeger.dispatcher.span_name'), [$command, $methodName]),
            $parentSpan->getContext()->getSpanId(),
            Reference::CHILD_OF
        );

        SpanTagHelper::setTags(
            $span,
            Caller::call(Config::get('jaeger.dispatcher.tags'), [$command])
        );

        optional($this->tracer->getScopeManager()->getActive())->close();
        $this->tracer->flush();

        $command->jaegerTracingContext = [];
        $this->tracer->inject($parentSpan->getContext(), Formats\TEXT_MAP, $command->jaegerTracingContext);

        return $command;
    }
}
