<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger;

use Exception;
use Illuminate\Console\Events\CommandFinished;
use Illuminate\Console\Events\CommandStarting;
use Illuminate\Contracts\Bus\Dispatcher;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Http\Kernel;
use Illuminate\Http\Client\Events\RequestSending;
use Illuminate\Log\Events\MessageLogged;
use Illuminate\Support\Facades\Queue;
use Iranserver\LaravelJaeger\Configurations\Guzzle\GuzzleHttpBinder;
use Iranserver\LaravelJaeger\Listeners\Guzzle\GuzzleListener;
use Iranserver\LaravelJaeger\Listeners\Job\JobFinishedListener;
use Iranserver\LaravelJaeger\Listeners\Job\JobStartedListener;
use Illuminate\Database\Events\QueryExecuted;
use Iranserver\LaravelJaeger\Listeners\Query\QueryListener;
use Illuminate\Queue\Events\JobProcessing;
use Illuminate\Support\Facades\Config as ConfigRepository;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Iranserver\LaravelJaeger\Middleware\HttpTracingMiddleware;
use Jaeger;
use Jaeger\Config;
use Jaeger\Reporter\InMemoryReporter;
use Jaeger\Sampler\ConstSampler;
use Jaeger\ScopeManager;
use OpenTracing\GlobalTracer;
use OpenTracing\Tracer;
use Iranserver\LaravelJaeger\Listeners\Console\ConsoleCommandFinishedListener;
use Iranserver\LaravelJaeger\Listeners\Console\ConsoleCommandStartedListener;
use Iranserver\LaravelJaeger\Services\Job\JobWithTracingInjectionDispatcher;
use Illuminate\Queue\Events\JobFailed;
use Illuminate\Queue\Events\JobProcessed;
use Iranserver\LaravelJaeger\Listeners\Log\LogListener;

class LaravelJaegerServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $config = __DIR__ . '/config/jaeger.php';

        $this->mergeConfigFrom($config, 'jaeger');

        $this->publishes([
            $config => base_path('config/jaeger.php'),
        ], 'jaeger-config');
    }

    /**
     * @throws Exception
     */
    public function boot(): void
    {
        if (!ConfigRepository::get('jaeger.enabled', false)) {
            $this->configureFakeTracer();

            return;
        }

        $this->configureTracer();

        $this->traceRoutes();
        $this->listenLogs();
        $this->listenQueries();
        $this->listenConsoleEvents();
        $this->traceGuzzle();
        $this->extendJobsDispatcher();
        $this->traceJobs();
    }

    private function configureFakeTracer(): void
    {
        $tracer = new class(
            'fake-tracer',
            new InMemoryReporter(),
            new ConstSampler(),
            true,
            null,
            new ScopeManager()) extends \Jaeger\Tracer {

            protected function getHostName()
            {
                return null;
            }
        };

        $this->app->instance(Tracer::class, $tracer);
    }

    /**
     * @throws Exception
     */
    private function configureTracer(): void
    {
        if ($tracerCallable = ConfigRepository::get('jaeger.custom_tracer_callable', null)) {
            $this->app->singleton(Tracer::class, $tracerCallable);

            return;
        }

        $config = new Config(
            [
                'sampler' => [
                    'type' => Jaeger\SAMPLER_TYPE_CONST,
                    'param' => true,
                ],
                "local_agent" => [
                    "reporting_host" => ConfigRepository::get('jaeger.agent_host', '127.0.0.1'),
                    "reporting_port" => ConfigRepository::get('jaeger.agent_port', 6832),
                ],
                'dispatch_mode' => Config::JAEGER_OVER_BINARY_UDP,
            ],
            ConfigRepository::get('jaeger.tracer_name', 'application')
        );

        $config->initializeTracer();

        $tracer = GlobalTracer::get();

        $this->app->instance(Tracer::class, $tracer);
    }

    /**
     * @throws BindingResolutionException
     */
    private function traceRoutes(): void
    {
        if (!$this->app->runningInConsole() && ConfigRepository::get('jaeger.routes_enabled', true)) {
            $this->app->make(Kernel::class)->prependMiddleware(HttpTracingMiddleware::class);
        }
    }

    private function listenLogs(): void
    {
        if (!ConfigRepository::get('jaeger.logs_enabled', true)) {
            return;
        }

        Event::listen(
            MessageLogged::class,
            ConfigRepository::get('jaeger.log.listener', LogListener::class)
        );
    }

    private function listenQueries(): void
    {
        if (!ConfigRepository::get('jaeger.query_logs', true)) {
            return;
        }

        Event::listen(
            QueryExecuted::class,
            ConfigRepository::get('jaeger.query.listener', QueryListener::class)
        );
    }

    private function listenConsoleEvents(): void
    {
        if (!ConfigRepository::get('jaeger.console_enabled', true)) {
            return;
        }

        Event::listen(
            CommandStarting::class,
            ConfigRepository::get('jaeger.console.listener_started', ConsoleCommandStartedListener::class)
        );

        Event::listen(
            CommandFinished::class,
            ConfigRepository::get('jaeger.console.listener_finished', ConsoleCommandFinishedListener::class)
        );
    }

    private function traceGuzzle(): void
    {
        if (!ConfigRepository::get('jaeger.guzzle_enabled', true)) {
            return;
        }

        GuzzleHttpBinder::bind($this->app);

        Event::listen(
            RequestSending::class,
            ConfigRepository::get('jaeger.guzzle.listener', GuzzleListener::class)
        );
    }

    /**
     * @throws BindingResolutionException
     */
    private function extendJobsDispatcher(): void
    {
        if (!ConfigRepository::get('jaeger.jobs_enabled', true)) {
            return;
        }

        $this->app->booted(function () {
            Queue::createPayloadUsing(function ($connectionName, $queue, $payload) {
                $command = isset($payload['data']['command']) ? (array)$payload['data']['command'] : [];
                return isset($command['jaegerTracingContext']) ? ['jaegerTracingContext' => $command['jaegerTracingContext']] : [];
            });
        });

        $dispatcher = $this->app->make(Dispatcher::class);
        $this->app->extend(Dispatcher::class, function () use ($dispatcher) {
            return $this->app->make(JobWithTracingInjectionDispatcher::class, [
                'dispatcher' => $dispatcher,
            ]);
        });
    }

    private function traceJobs(): void
    {
        if (!ConfigRepository::get('jaeger.jobs_enabled', true)) {
            return;
        }

        Event::listen(
            JobProcessing::class,
            ConfigRepository::get('jaeger.job.listener_started', JobStartedListener::class)
        );

        Event::listen(
            [JobProcessed::class, JobFailed::class],
            ConfigRepository::get('jaeger.job.listener_finished', JobFinishedListener::class)
        );
    }
}
