<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Services\Job;

use Illuminate\Contracts\Bus\QueueingDispatcher;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Queue\Events\JobQueued;
use Illuminate\Support\Facades\Config;
use Iranserver\LaravelJaeger\Services\Caller;
use Iranserver\LaravelJaeger\Services\Span\SpanTagHelper;
use OpenTracing\Tracer;

class JobWithTracingInjectionDispatcher implements QueueingDispatcher
{
    private $dispatcher;
    private $injectionMaker;

    public function __construct(QueueingDispatcher $dispatcher, JobInjectionMaker $injectionMaker, Tracer $tracer, Dispatcher $events)
    {
        $this->dispatcher = $dispatcher;
        $this->injectionMaker = $injectionMaker;

        $events->listen(JobQueued::class, function (JobQueued $event) use ($tracer) {
            $span = $tracer->getScopeManager()->getActive();
            if ($span) {
                SpanTagHelper::setTags($span->getSpan(), Caller::call(Config::get('jaeger.dispatcher.tags'), [$event]));
                optional($span)->close();
            }
        });
    }

    public function dispatchNow($command, $handler = null)
    {
        return $this->dispatcher->dispatchNow($this->injectionMaker->injectParentSpanToCommand($command), $handler);
    }

    public function dispatchToQueue($command)
    {
        return $this->dispatcher->dispatchToQueue($this->injectionMaker->injectParentSpanToCommand($command));
    }

    public function dispatch($command)
    {
        return $this->dispatcher->dispatch($this->injectionMaker->injectParentSpanToCommand($command));
    }

    public function dispatchSync($command, $handler = null)
    {
        return $this->dispatcher->dispatchSync($this->injectionMaker->injectParentSpanToCommand($command));
    }

    public function hasCommandHandler($command)
    {
        return $this->dispatcher->hasCommandHandler($command);
    }

    public function getCommandHandler($command)
    {
        return $this->dispatcher->getCommandHandler($command);
    }

    public function pipeThrough(array $pipes)
    {
        return $this->dispatcher->pipeThrough($pipes);
    }

    public function map(array $map)
    {
        return $this->dispatcher->map($map);
    }

    public function findBatch(string $batchId)
    {
        return $this->dispatcher->findBatch($batchId);
    }

    public function batch($jobs)
    {
        return $this->dispatcher->batch($jobs);
    }

    public function chain($jobs)
    {
        return $this->dispatcher->chain($jobs);
    }
}
