<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Middleware;

use Illuminate\Support\Facades\Config;
use OpenTracing\Reference;
use OpenTracing\Tracer;
use Symfony\Component\HttpFoundation\Response;
use Iranserver\LaravelJaeger\Services\Caller;
use Iranserver\LaravelJaeger\Services\Http\TracingRequestGuard;
use Iranserver\LaravelJaeger\Services\Span\ActiveSpanTraceIdRetriever;
use Iranserver\LaravelJaeger\Services\Span\SpanCreator;
use Iranserver\LaravelJaeger\Services\Span\SpanTagHelper;
use Iranserver\LaravelJaeger\Services\TraceIdHeaderRetriever;

class HttpTracingMiddleware
{
    private $tracer;
    private $spanCreator;
    private $requestGuard;
    private $activeTraceIdRetriever;
    private $traceIdHeaderRetriever;

    public function __construct(
        Tracer                     $tracer,
        SpanCreator                $spanCreator,
        TracingRequestGuard        $requestGuard,
        ActiveSpanTraceIdRetriever $activeTraceIdRetriever,
        TraceIdHeaderRetriever     $traceIdHeaderRetriever
    )
    {
        $this->tracer = $tracer;
        $this->spanCreator = $spanCreator;
        $this->requestGuard = $requestGuard;
        $this->activeTraceIdRetriever = $activeTraceIdRetriever;
        $this->traceIdHeaderRetriever = $traceIdHeaderRetriever;
    }

    public function handle($request, callable $next)
    {
        if (!$this->requestGuard->allowRequest($request)) {
            return $next($request);
        }

        $traceIdHeader = $this->traceIdHeaderRetriever->retrieve(iterator_to_array($request->headers));

        $this->spanCreator->create(
            Caller::call(Config::get('jaeger.http.span_name'), [$request]),
            $traceIdHeader,
            Reference::CHILD_OF
        );

        $response = $next($request);

        $this->addTraceIdToHeaderIfNeeded($response);

        return $response;
    }

    public function terminate($request, $response)
    {
        $scope = $this->tracer->getScopeManager()
            ->getActive();
        if (!$scope) {
            $this->tracer->flush();
            return;
        }

        $callableConfig = Config::get('jaeger.http.tags', function () {
            return [
                'type' => 'http',
            ];
        });

        SpanTagHelper::setTags($scope->getSpan(), Caller::call($callableConfig, [$request, $response]));

        $scope->close();
        $this->tracer->flush();
    }

    private function addTraceIdToHeaderIfNeeded(Response $response): void
    {
        $headerName = Config::get('jaeger.trace_id_header', null);
        if (!$headerName) {
            return;
        }

        $traceId = $this->activeTraceIdRetriever->retrieve();
        if (!$traceId) {
            return;
        }

        $response->headers->set($headerName, $traceId);
    }
}
