<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger;

use Illuminate\Console\Events\CommandFinished;
use Illuminate\Console\Events\CommandStarting;
use Illuminate\Contracts\Bus\Dispatcher;
use Illuminate\Log\Events\MessageLogged;
use Iranserver\LaravelJaeger\Listeners\JobFinishedListener;
use Iranserver\LaravelJaeger\Listeners\JobStartedListener;
use Illuminate\Database\Events\QueryExecuted;
use Iranserver\LaravelJaeger\Listeners\QueryListener;
use OpenTracing\Reference;
use Illuminate\Queue\Events\JobProcessing;
use Illuminate\Support\Facades\Config as ConfigRepository;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Jaeger;
use Jaeger\Config;
use Jaeger\Reporter\InMemoryReporter;
use Jaeger\Sampler\ConstSampler;
use Jaeger\ScopeManager;
use OpenTracing\GlobalTracer;
use OpenTracing\Tracer;
use Iranserver\LaravelJaeger\Listeners\ConsoleCommandFinishedListener;
use Iranserver\LaravelJaeger\Listeners\ConsoleCommandStartedListener;
use Iranserver\LaravelJaeger\Services\Job\JobWithTracingInjectionDispatcher;
use Illuminate\Queue\Events\JobFailed;
use Illuminate\Queue\Events\JobProcessed;

class LaravelJaegerServiceProvider extends ServiceProvider
{
    public function register()
    {
        $config = __DIR__ . '/config/jaeger.php';

        $this->mergeConfigFrom($config, 'jaeger');

        $this->publishes([
            $config => base_path('config/jaeger.php'),
        ], 'jaeger-config');
    }

    public function boot()
    {
        if (!ConfigRepository::get('jaeger.enabled', false)) {
            $this->configureFakeTracer();

            return;
        }

        $this->configureTracer();

        $this->listenLogs();
        $this->listenQueries();
        $this->listenConsoleEvents();
        $this->extendJobsDispatcher();

        $this->traceJobs();
    }

    public function configureFakeTracer(): void
    {
        $tracer = new class(
            'fake-tracer',
            new InMemoryReporter(),
            new ConstSampler(),
            true,
            null,
            new ScopeManager()) extends \Jaeger\Tracer {

            protected function getHostName()
            {
                return null;
            }
        };

        $this->app->instance(Tracer::class, $tracer);
    }

    public function extendJobsDispatcher(): void
    {
        $dispatcher = $this->app->make(Dispatcher::class);
        $this->app->extend(Dispatcher::class, function () use ($dispatcher) {
            return $this->app->make(JobWithTracingInjectionDispatcher::class, [
                'dispatcher' => $dispatcher,
            ]);
        });
    }

    private function configureTracer(): void
    {
        if ($tracerCallable = ConfigRepository::get('jaeger.custom_tracer_callable', null)) {
            $this->app->singleton(Tracer::class, $tracerCallable);

            return;
        }

        $config = new Config(
            [
                'sampler' => [
                    'type' => Jaeger\SAMPLER_TYPE_CONST,
                    'param' => true,
                ],
                "local_agent" => [
                    "reporting_host" => ConfigRepository::get('jaeger.agent_host', '127.0.0.1'),
                    "reporting_port" => ConfigRepository::get('jaeger.agent_port', 6832),
                ],
                'dispatch_mode' => Config::JAEGER_OVER_BINARY_UDP,
            ],
            ConfigRepository::get('jaeger.tracer_name', 'application')
        );

        $config->initializeTracer();

        $tracer = GlobalTracer::get();

        $this->app->instance(Tracer::class, $tracer);
    }

    private function listenLogs(): void
    {
        if (!ConfigRepository::get('jaeger.logs_enabled', true)) {
            return;
        }

        Event::listen(MessageLogged::class, function (MessageLogged $e) {
            $span = $this->app->make(Tracer::class)->getActiveSpan();
            if (!$span) {
                return;
            }

            if ($e->level == 'error')
                $span->setTag('error', true);

            $span->log([
                'message' => (string) $e->message,
                'context' => $e->context,
                'level' => $e->level,
            ]);
        });
    }

    private function listenQueries()
    {
        if (!ConfigRepository::get('jaeger.query_logs', true)) {
            return;
        }

        Event::listen(QueryExecuted::class, QueryListener::class);
    }

    private function listenConsoleEvents(): void
    {
        if (!$this->app->runningInConsole()) {
            return;
        }

        Event::listen(
            CommandStarting::class,
            ConfigRepository::get('jaeger.console.listeners.started', ConsoleCommandStartedListener::class)
        );

        Event::listen(
            CommandFinished::class,
            ConfigRepository::get('jaeger.console.listeners.finished', ConsoleCommandFinishedListener::class)
        );
    }

    private function traceJobs()
    {
        if (!$this->app->runningInConsole() or !ConfigRepository::get('jaeger.jobs_enabled', true)) {
            return;
        }

        Event::listen(
            JobProcessing::class,
            ConfigRepository::get('jaeger.job.listeners.started', JobStartedListener::class)
        );

        Event::listen(
            [JobProcessed::class, JobFailed::class],
            ConfigRepository::get('jaeger.job.listeners.finished', JobFinishedListener::class)
        );
    }
}
