<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Tests;

use GuzzleHttp\Psr7\Request as PsrRequest;
use Illuminate\Foundation\Application;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Jaeger\Reporter\InMemoryReporter;
use Jaeger\Sampler\ConstSampler;
use Jaeger\ScopeManager;
use Jaeger\Tracer;
use Orchestra\Testbench\TestCase;
use Iranserver\LaravelJaeger\Configurations\Http\SpanNameResolver;
use Iranserver\LaravelJaeger\LaravelJaegerServiceProvider;

abstract class JaravelTestCase extends TestCase
{
    protected $reporter;

    /**
     * @param Application $app
     */
    protected function defineEnvironment($app)
    {
        $this->reporter = new InMemoryReporter();
        $app['config']->set('jaravel', $this->jaravelConfiguration());
    }

    /**
     * @param Application $app
     * @return string[]
     */
    protected function getPackageProviders($app)
    {
        return [LaravelJaegerServiceProvider::class];
    }

    private function jaravelConfiguration(): array
    {
        return [
            'enabled' => true,
            'tracer_name' => 'application',
            'agent_host' => '127.0.0.1',
            'agent_port' => 6831,
            'trace_id_header' => 'X-Trace-Id',
            'logs_enabled' => true,

            'custom_tracer_callable' => function () {
                return new Tracer(
                    'test-tracer',
                    $this->reporter,
                    new ConstSampler(),
                    true,
                    null,
                    new ScopeManager()
                );
            },

            'http' => [
                'span_name' => SpanNameResolver::class,
                'tags' => function (Request $request, Response $response) {
                    return [
                        'type' => 'http',
                        'request_host' => $request->getHost(),
                        'request_path' => $request->path(),
                        'request_method' => $request->method(),
                        'response_status' => $response->getStatusCode(),
                        'error' => !$response->isSuccessful() && !$response->isRedirection(),
                    ];
                },
            ],

            'console' => [
                'span_name' => function (string $command) {
                    return 'Console: ' . $command;
                },
                'filter_commands' => ['schedule:run', 'horizon', 'queue:'],
                'tags' => function (string $command, int $exitCode) {
                    return [
                        'type' => 'console',
                        'console_command' => $command,
                        'console_exit_code' => $exitCode,
                    ];
                },
            ],

            'job' => [
                'span_name' => function ($realJob) {
                    return 'Job: ' . get_class($realJob);
                },
                'tags' => function ($realJob) {
                    return [
                        'type' => 'job',
                        'job_class' => get_class($realJob),
                    ];
                },
            ],

            'guzzle' => [
                'span_name' => function (PsrRequest $request) {
                    return 'request ' . $request->getUri()->getHost();
                },
                'tags' => function (PsrRequest $request) {
                    return [
                        'type' => 'request',
                        'uri' => (string)$request->getUri(),
                        'method' => $request->getMethod(),
                        'body' => $request->getBody()
                            ->getContents(),
                    ];
                },
            ],
        ];
    }
}
