<?php

declare(strict_types=1);

use Illuminate\Contracts\Queue\Job;
use Illuminate\Http\Request;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\HttpFoundation\Response;
use Iranserver\LaravelJaeger\Configurations;

return [

    /**
     * Enable JAEGER tracing or not. If not, noop tracer will be used.
     */
    'enabled' => env('JAEGER_ENABLED', true),

    /**
     * Name of your service, that will be shown in Jaeger panel
     */
    'tracer_name' => env('JAEGER_TRACER_NAME', 'application'),

    'agent_host' => env('JAEGER_AGENT_HOST', '127.0.0.1'),

    'agent_port' => env('JAEGER_AGENT_PORT', 6832),

    /**
     * Header name for trace`s id, that will be responded by TraceIdHttpHeaderMiddleware
     */
    'trace_id_header' => env('JAEGER_TRACE_ID_HEADER', 'x-trace-id'),

    /**
     * Every log in your application will be added to active span, if enabled
     */
    'logs_enabled' => env('JAEGER_LOGS_ENABLED', true),

    /**
     * Every log in your application will be added to active span, if enabled
     */
    'query_logs' => env('JAEGER_QUERY_LOGS', true),

    /**
     * Every jobs that process in your application will be traced
     */
    'jobs_enabled' => env('JAEGER_JOBS_ENABLED', true),

    /**
     * Describes configuration for incoming Http requests
     */
    'http' => [
        'span_name' => Configurations\Http\SpanNameResolver::class,
        'tags' => function (Request $request, Response $response) {
            return [
                'type'            => 'http',
                'request_host'    => $request->getHost(),
                'request_path'    => $request->path(),
                'request_method'  => $request->method(),
                'response_status' => $response->getStatusCode(),
                'error'           => !$response->isSuccessful() && !$response->isRedirection(),
                'env'             => env('APP_ENV')
            ];
        },
    ],

    /**
     * Describes configuration for console commands
     */
    'console' => [
        'span_name' => function (string $command, ?InputInterface $input = null) {
            return 'Console: ' . $command;
        },
        'filter_commands' => ['schedule:run', 'horizon', 'queue:'],
        'tags' => function (string $command, int $exitCode, ?InputInterface $input = null, ?OutputInterface $output = null) {
            return [
                'type'              => 'console',
                'console_command'   => $command,
                'console_exit_code' => $exitCode,
                'env'               => env('APP_ENV')
            ];
        },
    ],

    /**
     * Describes configuration for queued jobs
     */
    'job' => [
        'span_name' => function (?Job $job) {
            return 'Job: ' . optional($job)->resolveName();
        },
        'tags' => function (?Job $job) {
            return [
                'type' => 'job',
                'job_class' => optional($job)->resolveName(),
                'job_id' => optional($job)
                    ->getJobId(),
                'job_connection_name' => optional($job)
                    ->getConnectionName(),
                'job_name' => optional($job)
                    ->getName(),
                'job_queue' => optional($job)
                    ->getQueue(),
                'job_attempts' => optional($job)
                    ->attempts(),
                'error' => optional($job)->hasFailed(),
                'env'   => env('APP_ENV')
            ];
        },
    ],

    /**
     * Describes configuration for Guzzle requests if you`re using middleware created by HttpTracingMiddlewareFactory
     */
    'guzzle' => [
        'span_name' => Configurations\Guzzle\SpanNameResolver::class,
        'tags' => Configurations\Guzzle\TagsResolver::class,
    ],
];
