<?php

declare(strict_types=1);

namespace Iranserver\LaravelJaeger\Middleware;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Iranserver\LaravelJaeger\Services\Exceptions\CallerException;
use Iranserver\LaravelJaeger\Services\Span\ParentSpans;
use Iranserver\LaravelJaeger\Traits\JaegerFunctions;
use OpenTracing\Reference;
use OpenTracing\Span;
use OpenTracing\Tracer;
use Symfony\Component\HttpFoundation\Response;
use Iranserver\LaravelJaeger\Services\Caller;
use Iranserver\LaravelJaeger\Services\Http\TracingRequestGuard;
use Iranserver\LaravelJaeger\Services\Span\SpanCreator;
use Iranserver\LaravelJaeger\Services\Span\SpanTagHelper;

class HttpTracingMiddleware
{
    use JaegerFunctions;

    private $tracer;
    private $spanCreator;
    private $requestGuard;

    public function __construct(Tracer $tracer, SpanCreator $spanCreator, TracingRequestGuard $requestGuard)
    {
        $this->tracer = $tracer;
        $this->spanCreator = $spanCreator;
        $this->requestGuard = $requestGuard;
    }

    /**
     * @throws CallerException
     */
    public function handle(Request $request, callable $next)
    {
        if (!$this->requestGuard->allowRequest($request)) {
            return $next($request);
        }

        $this->loadParentSpans($request->header('X-JAEGER', ''));

        $span = $this->spanCreator->create(
            Caller::call(Config::get('jaeger.http.span_name'), [$request]),
            ParentSpans::get(),
            Reference::CHILD_OF
        );

        ParentSpans::add($span);

        $response = $next($request);

        $this->addTraceIdToHeaderIfNeeded($span, $response);

        return $response;
    }

    /**
     * @throws CallerException
     */
    public function terminate(Request $request, Response $response)
    {
        /** @var Span|null $httpSpan */
        $httpSpan = ParentSpans::get();
        if (!$httpSpan) {
            $this->tracer->flush();
            return;
        }

        SpanTagHelper::setTags(
            $httpSpan,
            Caller::call(Config::get('jaeger.http.tags'), [$request, $response])
        );

        if (!$response->isSuccessful() && !$response->isRedirection()) {
            $httpSpan->setTag('error', true);
        }

        $httpSpan->finish();
        $this->tracer->flush();

        ParentSpans::remove($httpSpan);
    }

    private function addTraceIdToHeaderIfNeeded(Span $span, Response $response): void
    {
        $headerName = Config::get('jaeger.trace_id_header');
        if (!$headerName) {
            return;
        }

        $response->headers->set($headerName, $span->getContext()->getTraceId());
    }
}
