<?php

namespace Iranserver\Middleware;

use Closure;
use Illuminate\Queue\Events\JobProcessed;
use Iranserver\Events\ServiceCustomAction\CreatingEvent;
use Iranserver\Events\ServiceCustomAction\FailedEvent;
use Iranserver\Events\ServiceCustomAction\SucceedEvent;
use Iranserver\Models\Base\Info;
use Illuminate\Queue\Events\JobFailed;
use Illuminate\Support\Facades\Event;
use Throwable;

class ServiceNotifierMiddleware
{
    private static bool $listenerRegistered = false;
    private static array $uuids = [];

    public function __construct()
    {
        if (!self::$listenerRegistered) {

            Event::listen(JobFailed::class, function (JobFailed $event) {
                $uuid = $this->getJobUuid($event->job);

                /**
                 * check for job to be traced or not
                 */
                if (!in_array($uuid, self::$uuids)) {
                    return;
                }
                $chainUuid = unserialize($event->job->payload()['data']['command'])->chainUuid ?? null;
                FailedEvent::dispatch($uuid, $event->job->payload()['displayName'] , $chainUuid , $event->exception->getMessage(), new Info());
            });

            Event::listen(JobProcessed::class, function (JobProcessed $event) {
                $uuid = $this->getJobUuid($event->job);

                /**
                 * Skip processing if the job:
                 * - is not being traced,
                 * - has failed, or
                 * - has been released.
                 */
                if (!in_array($uuid, self::$uuids)  || $event->job->hasFailed() || $event->job->isReleased()) {
                    return;
                }
                $chainUuid = unserialize($event->job->payload()['data']['command'])->chainUuid ?? null;
                $jobChainPosition = unserialize($event->job->payload()['data']['command'])->jobChainPosition ?? null;
                SucceedEvent::dispatch($uuid, $event->job->payload()['displayName'] , $chainUuid , $jobChainPosition, new Info());
            });

            self::$listenerRegistered = true;
        }
    }

    /**
     *  Process the job.
     *
     * @param Object $job
     * @param Closure $next
     * @return void
     * @throws Throwable
     */
    public function handle(object $job, Closure $next)
    {
        /**
         * add job to be traced
         */
        $uuid = $this->getJobUuid($job->job);
        self::$uuids[] = $uuid;

        if ($job->job->attempts() == 1) {
            $chainUuid = unserialize($job->job->payload()['data']['command'])->chainUuid ?? null;
            $jobChainPosition = unserialize($job->job->payload()['data']['command'])->jobChainPosition ?? null;

            CreatingEvent::dispatch($job->getServiceId(), $uuid, $job->job->payload()['displayName'], $job->getPayload() , $chainUuid , $jobChainPosition , new Info());
        }

        $next($job);
    }

    /**
     * Extract the UUID from the job payload.
     *
     * @param object $job
     * @return string|null
     */
    private function getJobUuid(object $job): ?string
    {
        return $job->payload()['uuid'] ?? $job->payload()['id'] ?? null;
    }
}
