<?php

namespace Iranserver\WebSockets;

use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcastNow;
use Illuminate\Translation\Translator;
use Illuminate\Validation\Validator;
use Illuminate\Validation\ValidationException;
use Iranserver\Base\BaseJob;
use Iranserver\Models\Interfaces\InfoInterface;
use Iranserver\Models\Interfaces\UserInterface;

abstract class WebSocket extends BaseJob implements ShouldBroadcastNow
{
    use InteractsWithSockets;

    protected UserInterface $user;
    protected array $payload;
    private Validator $validator;

    public function __construct(array $payload, UserInterface $user, InfoInterface $info)
    {
        $this->onConnection('rabbitmq');
        $this->onQueue('websocket.cmd.push');
        $this->payload = $payload;
        $this->user = $user;

        throw_if(!in_array($info->service_name, $this->getSenders()), new \Exception('Websocket Error:The sender is not valid'));
        $this->payloadValidation();
    }

    /**
     * Determine the display type
     * @return string
     */
    abstract protected function getType(): string;

    /**
     * Specify list of senders that are allowed to send this websocket
     *
     * @return array
     */
    abstract protected function getSenders(): array;

    /**
     * Get the validation rules for the payload.
     *
     * This method returns an array of validation rules that will be used
     * to validate the payload data. The rules specify the required fields
     * and their expected data types.
     *
     * @return array An associative array of validation rules, where the key
     *               is the field name and the value is the validation rule.
     */
    abstract protected function getPayloadRules(): array;

    /**
     * Validation of the sent payload
     *
     * @return void
     * @throws \Throwable
     */
    protected function payloadValidation()
    {
        $this->validator = new Validator(app(Translator::class), $this->payload, $this->getPayloadRules());
        throw_if($this->validator->fails(), new ValidationException($this->validator));
    }

    public function broadcastOn()
    {
        return new PrivateChannel("user-{$this->user->id}");
    }

    /**
     * Get the broadcast event name.
     *
     * This method allows you to define a custom name for the event when it is broadcasted.
     * The default name would be the class name of the event, but by overriding this method,
     * you can specify a custom event name to be used on the front end.
     *
     * @return string The name of the event when it is broadcasted.
     */
    public function broadcastAs(): string
    {
        return 'general';
    }

    /**
     * Get the data to broadcast.
     *
     * This method defines the structure of the data that will be broadcasted with the event.
     * By default, all public properties are broadcasted, but you can customize the payload
     * here by returning a specific array structure. This is useful to control exactly what
     * data gets sent to the front end listeners.
     *
     * @return array The data array to broadcast with the event.
     */
    public function broadcastWith()
    {
        $this->payloadValidation();
        return [
            'type' => $this->getType(),
            'data' => $this->validator->validated(),
        ];
    }
}