<?php

namespace IranServer\ERP\Drivers\Odoo;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class JsonRpcClient
{
    protected ?int $uid = null;
    protected Client $http;

    public function __construct(
        protected string      $database,
        protected string      $baseUrl,
        protected string      $username,
        protected string|null $token = null,
        protected string|null $password = null,
    )
    {
        $this->http = new Client([
            'base_uri' => $this->baseUrl,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'timeout' => 30,
        ]);
    }

    /**
     * Authenticate and cache UID
     */
    protected function authenticate(): int
    {
        if ($this->uid) {
            return $this->uid;
        }

        $payload = [
            'jsonrpc' => '2.0',
            'method' => 'call',
            'params' => [

                'service' => 'common',
                'method' => 'authenticate',
                'args' => [
                    $this->database,
                    $this->username,
                    $this->token,
                    ['interactive' => true]
                ],
            ],
            'id' => 1,
        ];

        $response = $this->send($payload);

        $uid = $response['result'] ?? null;
        if (!$uid) {
            throw new Exception('Failed to authenticate with Odoo.');
        }

        return $this->uid = $uid;
    }

    /**
     * Generic JSON-RPC call
     */
    protected function call(string $model, string $method, array $args = [], array $kwargs = [])
    {
        $uid = $this->authenticate();

        $payload = [
            'jsonrpc' => '2.0',
            'method' => 'call',
            'params' => [
                'service' => 'object',
                'method' => 'execute_kw',
                'args' => [
                    $this->database,
                    $uid,
                    $this->token,
                    $model,
                    $method,
                    $args,
                    $kwargs,
                ],
            ],
            'id' => 2,
        ];

        $response = $this->send($payload);

        if (isset($response['error'])) {
            throw new Exception('Odoo API error: ' . json_encode($response['error']));
        }

        return $response['result'] ?? null;
    }

    /**
     * Send a JSON-RPC request via Guzzle
     */
    protected function send(array $payload): array
    {
        try {
            $res = $this->http->post('/jsonrpc', ['json' => $payload]);
            $body = json_decode($res->getBody()->getContents(), true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid JSON response from Odoo.');
            }
            return $body;
        } catch (GuzzleException $e) {
            throw new Exception("HTTP request failed: " . $e->getMessage());
        }
    }

    // ---- Public helpers ----

    public function createRecord(string $model, array $values)
    {
        return $this->call($model, 'create', [$values]);
    }

    public function searchRead(string $model, array $domain = [], array $fields = [])
    {
        return $this->call($model, 'search_read', [$domain, $fields]);
    }

    public function updateRecord(string $model, int $id, array $values)
    {
        return $this->call($model, 'write', [[$id], $values]);
    }

    public function deleteRecord(string $model, int $id)
    {
        return $this->call($model, 'unlink', [[$id]]);
    }
}
