<?php

declare(strict_types=1);

namespace Iranserver\Basics\Helpers;

use Throwable;

final class Base64Url
{
    /**
     * Encodes a given string to Base64Url format.
     *
     * This method first encodes the input string into Base64 format using PHP's `base64_encode()` function,
     * then replaces the Base64 characters (`+` and `/`) with their URL-safe equivalents (`-` and `_`),
     * and finally removes any padding characters (`=`) from the encoded string.
     *
     * @param string $string The string to be encoded in Base64Url format.
     *
     * @return string The Base64Url-encoded string.
     *                Returns an empty string if an error occurs during encoding.
     */
    public static function encode(string $string): string
    {
        try {
            return rtrim(strtr(base64_encode($string), '+/', '-_'), '=');
        } catch (Throwable $throwable) {
            return '';
        }
    }

    /**
     * Decodes a given Base64Url-encoded string back to its original form.
     *
     * This method first converts the Base64Url-encoded string back to Base64 format by replacing URL-safe characters
     * (`-` and `_`) with their Base64 equivalents (`+` and `/`). It then pads the string to make its length a multiple of 4
     * (as required by Base64 encoding) and decodes it using PHP's `base64_decode()` function.
     *
     * @param string $string The Base64Url-encoded string to be decoded.
     *
     * @return string The decoded string.
     *                Returns an empty string if an error occurs during decoding.
     */
    public static function decode(string $string): string
    {
        try {
            return base64_decode(str_pad(strtr($string, '-_', '+/'), strlen($string) % 4, '=', STR_PAD_RIGHT));
        } catch (Throwable $throwable) {
            return '';
        }
    }
}