<?php

declare(strict_types=1);

namespace Iranserver\Basics\Helpers;

use Throwable;

final class Encryption
{
    private const CIPHER = "AES-256-CBC";

    /**
     * Encrypts a given string using the specified encryption key.
     *
     * This method uses OpenSSL to perform encryption on the provided string.
     * It generates a random initialization vector (IV) for each encryption process and combines it with the encrypted string before encoding the result in Base64Url format.
     *
     * @param string $string The string to be encrypted.
     * @param string $key The key used for encryption. It must be a valid encryption key.
     *
     * @return string The Base64Url-encoded encrypted string with the IV appended.
     *                Returns an empty string in case of an error or exception.
     */
    public static function encrypt(string $string, string $key): string
    {
        try {
            $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length(self::CIPHER));

            $encryptedData = openssl_encrypt($string, self::CIPHER, $key, 0, $iv);

            return Base64Url::encode($encryptedData . '::' . $iv);
        } catch (Throwable $throwable) {
            return '';
        }
    }

    /**
     * Decrypts a given encrypted string using the specified decryption key.
     *
     * This method first decodes the provided string from Base64Url format, then separates the
     * encrypted data and the initialization vector (IV) which are stored together in the string.
     * The method uses OpenSSL to perform the decryption process with the given key and IV.
     *
     * @param string $string The encrypted string (in Base64Url format) that needs to be decrypted.
     * @param string $key The key used for decryption. It must match the key used during encryption.
     *
     * @return string The decrypted string, or an empty string if the decryption fails or an error occurs.
     */
    public static function decrypt(string $string, string $key): string
    {
        try {
            list($encryptedData, $iv) = explode('::', Base64Url::decode($string));

            return openssl_decrypt($encryptedData, self::CIPHER, $key, 0, $iv);
        } catch (Throwable $throwable) {
            return '';
        }
    }
}