<?php

namespace Iranserver\Basics\Ordering\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Iranserver\Basics\Ordering\Resources\OrderTrackingResource;

/**
 * Class OrderTracking
 * @package Iranserver\Basics\Ordering\Models
 *
 * @property int $id
 * @property Carbon $created_at
 * @property array $payload
 * @property Carbon $finished_at
 * @property Carbon $prune_at
 *
 * @method self|Builder garbage()
 * @method static self|Builder query()
 */
class OrderTracking extends Model
{
    use HasFactory;

    protected static $pruneAfterDays = 14;

    protected $table = "order_tracking";
    protected $fillable = ['payload', 'finished_at', 'prune_at'];
    protected $casts    = ['payload' => 'array', 'finished_at' => 'datetime', 'prune_at' => 'datetime'];

    public function getUpdatedAtColumn()
    {
        return null;
    }

    public static function pruneAfterDays($days)
    {
        self::$pruneAfterDays = $days;
    }

    public static function boot()
    {
        parent::boot();
        static::saving(function($order) {
            if(!isset($order->attributes['prune_at']))  {
                $order->attributes['prune_at'] = now()->addDays(self::$pruneAfterDays);
            }
        });
    }

    public function scopeGarbage($query)
    {
        $query->where('prune_at', '<=', now());
    }

    /**
     * @param array $payload
     * @return OrderTracking
     */
    public function finish()
    {
        $payload = func_get_args();
        return $this->fill([
            'finished_at' => now(),
            'payload' => $payload ? $payload[0] : $this->payload
        ]);
    }

    /**
     * @return OrderTrackingResource
     */
    public function resource()
    {
        return new OrderTrackingResource($this);
    }
}
