<?php

namespace Iranserver\Basics\Primary\Sync;

use Carbon\Carbon;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\Cache;
use InvalidArgumentException;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Iranserver\Metrics\Metrics;
use Throwable;

/**
 * همگام سازی دیتابیس با هاب
 *
 * این توسعه به شما امکانی میدهد که بتوانید مدل های جاری سیستم را با دیتابیس هاب بروز نگهدارید.
 * فقط کافیست یک command لاراول ایجاد کرده و از نوع SyncTableCommand قرار دهید
 * در ادامه منطق سینک مربوط به model خود را بنویسید.
 *
 *
 * | متغیر | توضیحات |
 * | ----------- | ----------- |
 * | updated_at | مشخص کننده نام ستونی هست که آخرین بار تا چه رکوردی سینک انجام شده |
 * | table | مشخص کننده نام جدولی هست که دیتا از آن لود میشوند |
 * | connection | مشخص کننده نام کانکشن دیتابیس میباشد |
 *
 * مثال:
 *
 * ```php
 * namespace App\Console\Commands;
 *
 * use stdClass;
 *
 * class YourSyncCommand extends SyncTableCommand
 * {
 *     protected $signature = 'app:your-sync-command';
 *
 *     protected string $table = "source_table";
 *
 *     public function updateOrCreate(stdClass $record): void
 *     {
 *         // save or update your model (do not trigger pub/sub events)
 *         // this method support service injection
 *     }
 * }
 * ```
 *
 * چنانچه میخواهید کوئری لود دیتا رو تغییر دهید، متد زیر را تغییر دهید:
 *
 * ```php
 * public function query(): Builder
 * ```
 *
 * > توجه داشته باشید که پس از ذخیره یا آپدیت مدل خود نیازی نیست هیچ event سراری را اجرا کنید.
 * مثلا: اگر رکوردی در جدول سرویس ها آپدیت شد، نیازی نیست از طریق صف به سایر سرویس ها اطلاع داده شود و این کار
 * می بایست بصورت silent انجام شود.
 *
 * جهت مشاهده جزئیات command از دستور زیر استفاده کنید:
 *
 * `php artisan app:your-sync-command --help`
 *
 * ```shell
 * --dry-run            Perform a dry run without syncing any data
 *  -c, --continue-on-error  Do not exit sync on errors
 * --query[=QUERY]      Sync data updated within the specified period of time. (values: modified, all, 1m, 1d, ...) [default: "modified"]
 * --chunk[=CHUNK]      Number of records to sync per chunk [default: "200"]
 * --count[=COUNT]      Maximum number of records to sync per run [default: "all"]
 * --rest[=REST]        Number of seconds to rest between updates [default: "0"]
 * ```
 *
 * نمای کلی سینک:
 *
 * ![sync](resources/doc-sync.png)
 *
 * @package Sync
 */
abstract class SyncTableCommand extends Command
{
    protected string $baseSignature =
        '{--dry-run : Perform a dry run without syncing any data}
         {--c|continue-on-error : Do not exit sync on errors}
         {--query=modified : Sync data updated within the specified period of time. (values: modified, all, 1m, 1d, ...)}
         {--chunk=200 : Number of records to sync per chunk}
         {--count=all : Maximum number of records to sync per run}
         {--rest=0 : Number of seconds to rest between updates}';

    protected $description = 'Sync data from a remote table to the microservice.';

    protected string $connection = 'hub';

    protected string $table = '';

    protected string $updated_at = 'updated_at';

    public function __construct()
    {
        $this->signature .= PHP_EOL . $this->baseSignature;
        parent::__construct();
    }

    public function handle(): void
    {
        // todo: isolate with jobs
        $dryRun = $this->option('dry-run');
        $chunk  = $this->option('chunk');
        $count  = $this->option('count');
        $rest   = $this->option('rest');
        $continueOnError = $this->option('continue-on-error');

        if (!$this->table) {
            throw new InvalidArgumentException("Table name can not be empty.");
        }

        [$successfulSync, $failedSync] = $this->metrics();

        //todo: fix Allowed memory size of 2147483648 bytes exhausted

        $this->query()->each(
            function ($record) use ($dryRun, $rest, $successfulSync, $failedSync, $continueOnError, &$count) {

                if (!$dryRun) {
                    try {
                        app()->call([$this, 'updateOrCreate'], ['record' => $record]);
                        $successfulSync->inc([class_basename(static::class)]);
                        $this->info(sprintf("Record [%s] is synced", $record->id));
                        $this->lastRecordPointer($record->{$this->updated_at});
                    } catch (Throwable $exception) {
                        $failedSync->inc([class_basename(static::class)]);
                        //todo: keep failed --failed
                        $this->warn(sprintf("Record %s has an error: %s", $record->id, $exception->getMessage()));
                        throw_unless($continueOnError, $exception);
                    }
                }
                else $this->info(sprintf("Record [%s] is synced", $record->id));
                if ($count != 'all' && --$count <= 0) return false;
                sleep($rest);
                return true;

            },
            $chunk
        );
    }

    public function query(): Builder
    {
        $time = $this->option('query');

        return DB::connection($this->connection)->table($this->table)
            ->where(function ($query) use ($time) {
                return match ($time) {
                    "all" => $query, // don't touch query
                    "modified" => $query->where($this->updated_at, '>', $this->lastRecordPointer()),
                    default => $query->where($this->updated_at, '>', Carbon::now()->sub($time)->toDateTimeString())
                };
            })
            ->orderBy($this->updated_at);
    }

    /**
     * @return array list of metrics.
     */
    public function metrics(): array
    {
        return [
            Metrics::getOrRegisterCounter(
                'application', 'successful_sync_count',
                'Number successful records',
                ['name']
            ),
            Metrics::getOrRegisterCounter(
                'application', 'failed_sync_count',
                'Number failed records',
                ['name']
            )
        ];
    }

    /**
     * When a sync schedule run successfully, we save the last pointer to
     * avoid processing it again until the next record is modified.
     *
     * @param string|null $value set if you want to store pointer
     * @return string last pointer
     */
    public function lastRecordPointer(string $value = null): string
    {
        $key = static::class . '-last-record-pointer';
        if (filled($value)) {
            Cache::set($key, $value);
            return $value;
        }

        return Cache::get($key, '0000-00-00');
    }
}
