<?php

namespace is\core\components;

use is\core\BaseApp;
use is\core\shared\Arr;
use is\core\shared\Str;
use is\core\shared\Sys;
use ReflectionClass;
use Exception;

/**
 * Trait Core
 * @package is\core\components
 *
 * @mixin BaseApp
 */
trait Core
{
    private static $instances = [];
    protected $config = [];
    abstract public function config(): array;

    /** @return static */
    public static function instance(): self
    {
        if (isset(self::$instances[static::class])) {
            return self::$instances[static::class];
        }
        return self::$instances[static::class] = new static();
    }

    public function get(string $key, $default = null)
    {
        return Arr::get($this->config, $key, $default);
    }

    public function set(string $key, $value)
    {
        return Arr::set($this->config, $key, $value);
    }

    public function getRootPath($path = ''): string
    {
        $path = ltrim($path, "/\\");
        $appClass = new ReflectionClass($this);
        return Str::normalizePath(dirname($appClass->getFileName())."/$path");
    }

    public function run(): void
    {
        $this->startTracing();

        $route = $this->findRoute();

        if (is_callable($route[0])) {
            $response = $route[0]($this);
        }
        else {
            [$controller, $action] = $route;
            $controller = new $controller($this);
            $response = $controller->$action();
        }

        if (is_array($response)) $this->response->json($response);
        else if (is_string($response)) $this->response->raw($response);

        $this->callTraitsMethod('rendering');
        echo $this->response->render();
        $this->callTraitsMethod('terminate');
    }

    public function environment($env): bool
    {
        $current = $this->get('env', 'production');
        return strpos($current, $env) === 0;
    }

    public function runningInConsole(): bool
    {
        return php_sapi_name() === 'cli' or php_sapi_name() === 'phpdbg';
    }

    public function setVars($vars): self
    {
        if (is_array($vars)) $this->config['vars'] = array_merge($this->config['vars'] ?? [], $vars);
        else $this->config['vars'][$vars] = func_get_args()[1] ?? null;
        return $this;
    }

    public function uses($trait): bool
    {
        return in_array($trait, Sys::classUsesRecursive($this));
    }

    public function whenUses($trait, callable $call)
    {
        return $this->uses($trait) ? $call($this) : null;
    }

    public function whenNotUses($trait, callable $call)
    {
        return $this->uses($trait) ? null : $call($this);
    }

    public function shouldUse($trait)
    {
        $this->whenNotUses($trait, function () use ($trait) {
            throw new Exception(sprintf(
                "Trait (%s) should be used for (%s)",
                $trait, static::class
            ));
        });
    }

    private function startTracing()
    {
	    if (function_exists('startActiveSpan')) {
            $spanName = $_SERVER['REQUEST_METHOD'] . ': ' . ($_GET['action'] ?? 'Dashboard');
            startActiveSpan($spanName);
        }
    }
}
