<?php
declare(strict_types=1);

use PHPUnit\Framework\TestCase;
use is\core\shared\Arr;

final class ArrTest extends TestCase
{
    public function provider1(): array
    {
        return [
            'on array' => [
                [
                    'array' => true,
                    'users' => [
                        ['name' => 'foo', 'age' => 20, 'kids' => ['aa', 'bb'], 'country' => 'ca'],
                        ['name' => 'bar', 'age' => 30, 'kids' => ['cc', 'dd']]
                    ],
                    'anything',
                    'meta' => [
                        'total' => 2
                    ]
                ]
            ],
            'on object' => [
                (object)[
                    'array' => true,
                    'users' => [
                        (object)['name' => 'foo', 'age' => 20, 'kids' => ['aa', 'bb'], 'country' => 'ca'],
                        (object)['name' => 'bar', 'age' => 30, 'kids' => ['cc', 'dd']]
                    ],
                    'anything',
                    'meta' => (object)[
                        'total' => 2
                    ]
                ]
            ]
        ];
    }

    /**
     * @dataProvider provider1
     */
    public function testCommonGetBehaviors($target): void
    {
        $this->assertEquals($target, Arr::get($target));
        $this->assertEquals(null, Arr::get($target, ''));
        $this->assertEquals(false, Arr::get($target, 'object', false));
        $this->assertEquals(2, Arr::get($target, 'meta.sum', 2));
        $this->assertEquals(true, Arr::get($target, 'array'));
        $this->assertEquals(20, Arr::get($target, 'users.0.age'));
        $this->assertEquals(2, Arr::get($target, 'meta.total'));
        $this->assertEquals($target, Arr::get($target, '*'));
        $this->assertEquals(['foo', 'bar'], Arr::get($target, 'users.*.name'));
        $this->assertEquals('default', Arr::get($target, 'users.*.', 'default'));
        $this->assertEquals(['ca'], Arr::get($target, 'users.*.country'));
        $this->assertEquals(['bb', 'dd'], Arr::get($target, 'users.*.kids.1'));
        $this->assertEquals(
            is_array($target) ? $target['users'] : $target->users,
            Arr::get($target, 'users.*.*')
        );
    }

    public function provider2(): array
    {
        $r1 = $r2 = $r3 = $r4 = $r5 = $r6 = $r7 = $target = [
            'foo' => 'bar',
            'pop' => [
                'test' => [1, 2, 125],
                'test1' => [[5, 6], [6]],
                'hex' => ['blue' => '#0000FF']
            ],
            'apple'
        ];

        $r1['foo'] = 255;
        $r2['pop']['test'] = 255;
        $r3['pop']['test'] = [255, 255, 255];
        $r4[0] = ['apple'];
        $r5['pop']['test1'] = [[10, 10], [10]];
        $r6['pop']['test1'] = [[9, 6], [9]];
        $r7['pop']['test1'] = [[5, 8], [6, 8]];

        return [
            [null, 'foo', 255, null],
            [null, '', 255, null],
            [null, 'foo.bar', 255, null],
            ['name', 'foo', 255, 'name'],
            [[], 'foo', 255, ['foo' => 255]],
            [[], 'foo.*', 255, ['foo' => []]],
            [[], '', 255, ['' => 255]],
            [[], '*.foo', 255, []],
            [$target, 'foo', 255, $r1],
            [$target, 'pop.test', 255, $r2],
            [$target, 'pop.test.*', 255, $r3],
            [$target, '0', ['apple'], $r4],
            [$target, 'pop.test1.*.*', 10, $r5],
            [$target, 'pop.test1.*.0', 9, $r6],
            [$target, 'pop.test1.*.1', 8, $r7],
        ];
    }

    /**
     * @dataProvider provider2
     */
    public function testCommonSetBehaviors($target, $key, $value, $result): void
    {
        Arr::set($target, $key, $value);
        $this->assertEquals($result, $target);
    }

    public function testCommonHasBehaviors(): void
    {
        $target = [
            'foo' => 125,
            'x' => []
        ];
        $this->assertTrue(Arr::has($target, 'foo'));
        $this->assertTrue(Arr::has($target, 'x'));
        $this->assertFalse(Arr::has($target, 'bar'));
        $this->assertFalse(Arr::has($target, 'x.a'));
    }
}