<?php
namespace greenweb\core\models;

use Illuminate\Database\Capsule\Manager as Capsule;

class DbModel extends Model
{
    /**
     * @var array List of attributes which you don't want to save in database.
     */
    protected $skipAttributes = [];

    /**
     * @return bool|int Delete populated object.
     */
    public function delete()
    {
        if ($this->isPopulated) {
            $obj = static::find();
            foreach (static::primaryKeys() as $key) {
                $obj->where($key, '=', $this->oldAttributes[$key]);
            }
            return $obj->delete();
        }

        return false;
    }

    /**
     * @param bool|true $validate Set if you want to validate data.
     * @return bool True if record inserted or updated.
     */
    public function save($validate = true)
    {
        if ($validate and !$this->validate()) {
            return false;
        }

        $con = Capsule::connection();
        $con->beginTransaction();

        if (!$this->beforeSave(!$this->isPopulated)) {
            $con->rollBack();
            return false;
        }

        $data = array_diff_key($this->data(), array_flip($this->skipAttributes));

        if ($this->isPopulated) {
            if ($data != array_intersect_key($this->oldAttributes, $data)) {
                $obj = static::find();
                foreach (static::primaryKeys() as $key) {
                    $obj->where($key, '=', $this->oldAttributes[$key]);
                }
                $res = $obj->update($data);
            } else {
                $res = true;
            }
        } else {
            $res = static::find()->insert($data);
        }

        if ($res and $this->afterSave(!$this->isPopulated)) {
            $con->commit();
            return true;
        } else {
            $con->rollBack();
            return false;
        }
    }

    public function lastId()
    {
        return Capsule::connection()->getPdo()->lastInsertId();
    }

    /**
     * @param bool $insert Represent for insert or update.
     * @return bool Return false you want to stop save function.
     */
    public function beforeSave($insert)
    {
        return true;
    }

    /**
     * @param bool $insert Represent for insert or update.
     * @return bool Return false if you want to rollback transaction. Yep, we provide transaction for you ;)
     */
    public function afterSave($insert)
    {
        return true;
    }

    public static function find($alias = null)
    {
        return Capsule::table(static::tableName() . ($alias ? " AS $alias" : ""));
    }

    /**
     * @param $pk array|string Primary key.
     * @return static
     */
    public static function findOne($pk)
    {
        $result = static::find();
        if (!is_array($pk)) {
            $key = @static::primaryKeys()[0];
            $result->where($key, '=', $pk);
        } else foreach($pk as $key => $value) {
            $result->where($key, '=', $value);
        }
        $result = (array)$result->first();
        if (!$result) {
            return null;
        }
        return static::populate($result);
    }

    protected $isPopulated = false;
    protected $oldAttributes = [];

    public static function populate($data)
    {
        $instance = new static();
        $instance->oldAttributes = $data;
        $instance->load($data, true);
        $instance->isPopulated = true;
        return $instance;
    }

    public static $primaryKeys;

    public static function primaryKeys()
    {
        if(static::$primaryKeys === null) {
            $data = Capsule::connection()->select("DESCRIBE " . static::tableName());
            foreach ($data as $row) {
                if (strtoupper($row->Key) == 'PRI') {
                    static::$primaryKeys[] = $row->Field;
                }
            }
        }
        return static::$primaryKeys;
    }

    /**
     * @throws \Exception if not override this method.
     * @return string Table name.
     */
    public static function tableName()
    {
        throw new \Exception("Table name is not declared.");
    }
}