<?php
/**
 * @author Pooya <eraghi@greenweb.ir>
 * @licence MIT
 */
namespace greenweb\core\drawing;

use greenweb\core\base\PrimaryObject;
use phpooya\gdfarsitext\Bidi;

class Canvas extends PrimaryObject
{
    public $width;

    public $height;

    public $dir = 'rtl';

    public $isPersian = true;

    public $img;

    public function init()
    {
        $this->img = imagecreatetruecolor($this->width, $this->height);
    }

    /**
     * @param array $data Config of table.
     * @return Table object which created.
     */
    public function table(array $data = [])
    {
        $table = new Table(array_merge($data, ['img' => $this]));
        $table->render();
        return $table;
    }

    public function text($text, $size, $x, $y, $font, $color, $rotate = 0)
    {
        $y = $this->h($y);
        list($renderWidth ,, $text) = $return = $this->prepareText($size, $rotate, $font, $text);

        /** @TODO: text top is not exactly precise */
        imagefttext($this->img, $size, $rotate, $this->rtlX($x, $renderWidth), $y, $color, $font, $text);
        return $return;
    }

    public function prepareText($size, $rotate, $font, $text)
    {
        if ($this->isPersian) {
            $bidi = new Bidi();
            /** @var array $strArray */
            $strArray = $bidi->utf8Bidi($bidi->UTF8StringToArray($text), 'R');
            $text = '';
            foreach ($strArray as $chr) {
                $text .= $bidi->unichr($chr);
            }
        }

        $data = imagettfbbox($size, $rotate, $font, $text);
        return [
            abs($data[4] - $data[0]),
            abs($data[5] - $data[1]),
            $text
        ];
    }

    public function attach($filename, $x, $y, $w = null, $h = null)
    {
        $y = $this->h($y);
        $w = $this->w($w);
        $h = $this->h($h);
        $img = $this->getImage($filename);
        $imgW = imagesx($img);
        $imgH = imagesy($img);
        imagecopyresampled($this->img, $img, $this->rtlX($x, $w?:$imgW), $y, 0, 0, $w?:$imgW, $h?:$imgH, $imgW, $imgH);
        return [$img, $imgW, $imgH];
    }

    public function rectangle($x, $y, $w, $h, $color, $filled = true)
    {
        $y = $this->h($y);
        $w = $this->w($w);
        $h = $this->h($h);
        $x = $this->rtlX($x, $w);
        if($filled) {
            imagefilledrectangle($this->img, $x, $y, $x + $w, $y + $h, $color);
        } else {
            imagerectangle($this->img, $x, $y, $x + $w, $y + $h, $color);
        }
    }

    public function getImage($filename)
    {
        switch (getimagesize($filename)['mime']) {
            case "image/gif":
                return imagecreatefromgif($filename);
            case "image/png":
                return imagecreatefrompng($filename);
            default:
                return imagecreatefromjpeg($filename);
        }
    }

    public function color($r, $g, $b, $a = null)
    {
        if ($a === null) {
            return imagecolorexact($this->img, $r, $g, $b);
        } else {
            return imagecolorexactalpha($this->img, $r, $g, $b, $a);
        }
    }

    public function isRtl()
    {
        return strtolower($this->dir) === 'rtl';
    }

    public function rtlX($x, $width = 0)
    {
        $x = $this->w($x);
        $width = $this->w($width);
        return $this->isRtl() ? $this->width - $x - $width : $x;
    }

    public function w($x, $img = null)
    {
        return $this->toPixel($img === null ? $this->img : $img, $x, true);
    }

    public function h($x, $img = null)
    {
        return $this->toPixel($img === null ? $this->img : $img, $x, false);
    }

    public function toPixel($img, $x, $width)
    {
        if(substr($x, -1) === '%') {
            $x = substr($x, 0, -1);
            $x = ($x * ($width ? imagesx($img) : imagesy($img))) / 100;
        }
        return $x;
    }

    public function render()
    {
        header("content-type: image/png");
        imagepng($this->img);
    }
}