<?php
namespace greenweb\core\base;

use greenweb\core\App;
use greenweb\core\helpers\Helper;
use Whoops\Handler\CallbackHandler;
use Whoops\Handler\PrettyPageHandler;
use Whoops\Run;

class Debug extends Component
{
    const ALL = 'all';
    const ERROR = 'error';
    const INFO = 'info';
    const DEBUG = 'debug';
    const PROFILING = 'profiling';
    const JSLOG = 'jslog';
    const HANDLER = 'handler';

    public $dir;

    public $perPage = 20;

    /**
     * @var array List of errors you want to save.
     */
    public $level = [self::ALL];

    /**
     * @var array List of all errors in the request.
     */
    private $errors = [];

    public function init()
    {
        parent::init();
        if ($this->dir === null and !($this->app instanceof App)) {
            throw new \Exception("Property 'dir' can not be null.");
        }
        elseif ($this->dir === null) {
            $this->dir = $this->app->rootDir() . "/runtime/debug";
        }
    }

    private function log($type, $data)
    {
        if (in_array(self::ALL, $this->level) or in_array($type, $this->level)) {
            $time = microtime(true);
            $this->errors[] = [
                'type' => strtolower($type),
                'time' => $time,
                'data' => $data,
                'backtrace' => debug_backtrace(0)
            ];
        }
    }

    private function save()
    {
        if ($this->errors) {
            @ mkdir($this->dir, 0770, true);
            $time = microtime(true);
            $content = [
                'method' => $this->app->request->method(),
                'time' => $time,
                'errors' => $this->errors,
                'session' => $_SESSION,
                'server' => $_SERVER,
                'post' => $_POST,
                'get' => $_GET
            ];
            file_put_contents($this->dir . '/' . $time, json_encode($content));
            $this->errors = [];
        }
    }

    public function debug($data)
    {
        $this->log(self::DEBUG, $data);
    }

    public function error($data)
    {
        $this->log(self::ERROR, $data);
    }

    public function info($data)
    {
        $this->log(self::INFO, $data);
    }

    private $profilingTime = [];

    public function beginProfiling($text)
    {
        array_push($this->profilingTime, [
            'text' => $text,
            'duration' => Helper::microTime()
        ]);
    }

    public function endProfiling()
    {
        if($this->profilingTime) {
            $data = array_pop($this->profilingTime);
            $data['duration'] = Helper::microTime() - $data['duration'];
            $this->log(self::PROFILING, $data);
        }
    }

    /**
     * @param array $data
     */
    public function jsLog($data)
    {
        if(in_array(self::ALL, $this->level) or in_array(self::JSLOG, $this->level)) {
            ob_start(function ($output) use ($data) {
                $data = json_encode($data);
                return $output . "<script> var _log_data = {$data}; console.log(_log_data); </script>";
            });
        }
    }

    public function getData($page = 1)
    {
        if($page < 1) $page = 1;
        $p = ($page - 1) * $this->perPage;

        $files = @scandir($this->dir, SORT_DESC);
        $data = [];
        $i = 0;
        if ($files) foreach($files as $name) {
            if($p) {
                $p--;
                continue;
            }
            if($i >= $this->perPage) break;
            if($name == '.' or $name == '..') continue;
            $data[] = @ json_decode(file_get_contents($this->dir . "/$name"), true);
            $i++;
        }
        return $data;
    }

    public function registerDebugHandler()
    {
        $whoops = new Run();
        //$whoops->pushHandler(new PrettyPageHandler()); /** @TODO: find a way to use it internally. */
        $whoops->pushHandler(new CallbackHandler(function ($exception, $inspector, $run) {
            /** @var \Throwable $exception */
            $this->log(self::HANDLER, [
                'code' => $exception->getCode(),
                'file' => $exception->getFile(),
                'line' => $exception->getLine(),
                'message' => $exception->getMessage(),
                'trace' => $exception->getTraceAsString()
            ]);
        }));
        $whoops->register();
    }

    public function terminate()
    {
        $this->save();
    }
}