<?php

/**
 * @author Pooya <eraghi@greenweb.ir>
 * @licence MIT
 */

namespace greenweb\core;

use greenweb\core\base\Formatter;
use greenweb\core\base\Mailer;
use greenweb\core\events\BaseEventData;
use greenweb\core\events\EventsTrait;
use greenweb\core\helpers\Helper;
use greenweb\core\base\Request;
use greenweb\core\base\Session;
use greenweb\core\base\Debug;

abstract class App
{
    use EventsTrait;
    
    const EVENT_INIT = 'init';
    const NO_COMPONENT = 'no-component';

    /**
     * @var Request
     */
    public $request;

    /**
     * @var Session
     */
    public $session;

    /**
     * @var Formatter
     */
    public $formatter;

    /**
     * @var Debug
     */
    public $debug;

    /**
     * @var Mailer
     */
    public $mailer;

    public static $classMap = [];

    public $errors;

    public $errorsLevel = E_ALL | E_STRICT;

    public $config = [];

    public function __construct($config = [])
    {
        $properties = array_flip(Helper::getPublicProperties($this));
        unset($properties['config']);
        static::configure($this, array_intersect_key($config, $properties));
        $this->config = array_diff_key($config, $properties);
        if($this->errors !== null) {
            ini_set('display_errors', $this->errors);
            error_reporting($this->errors ? $this->errorsLevel : 0);
        }
        static::$instance = $this;
        $this->init();
    }

    public function beforeInit()
    {

    }

    public function init()
    {
        $this->beforeInit();
        
        foreach($this->coreComponent() as $key => $conf) {
            if ($this->{$key} !== static::NO_COMPONENT) {
                $this->{$key} = static::createObject(empty($this->{$key}) ? $conf : $this->{$key}, ['app' => $this]);
            }
        }
        $this->trigger(static::EVENT_INIT, new BaseEventData());
        
        $this->afterInit();
    }

    public function afterInit()
    {

    }

    public function coreComponent()
    {
        return [
             'request' => Request::className(),
             'session' => Session::className(),
               'debug' => Debug::className(),
           'formatter' => Formatter::className(),
              'mailer' => Mailer::className()
        ];
    }

    public static function autoload($class)
    {
        static::$classMap['greenweb\\core\\'] = __DIR__;

        foreach(static::$classMap as $ns => $dir) {
            if (strpos($class, $ns) === 0) {
                $className = str_replace('\\', '/', substr($class, strlen($ns)));
                greenRequireFile($dir . "/$className.php");
                break;
            }
        }
    }

    public static function createObject($options = [], $defaults = [])
    {
        if(empty($options)) {
            return null;
        } elseif(is_object($options)) {
            return $options;
        } elseif(is_string($options)) {
            $class = $options;
            $options = $defaults;
        } else {
            $class = @$options['class'];
            $options = array_merge($defaults, $options);
            unset($options['class']);
        }
        return new $class($options);
    }

    public static function configure($object, $config = [])
    {
        foreach($config as $attr => $value) {
            $object->{$attr} = $value;
        }
    }

    public static function rootDir()
    {
        return __DIR__;
    }

    public static function requirements()
    {
        return [
            'php-version' => function_exists('version_compare') and version_compare(PHP_VERSION, '5.0.0', '>=')
        ];
    }
}

spl_autoload_register(['greenweb\core\App', 'autoload'], false);

function greenRequireFile($file)
{
    require $file;
}