<?php
namespace greenweb\core\whmcs;

use greenweb\core\App;
use greenweb\core\cache\ArrayCache;
use greenweb\core\helpers\FileHelper;
use greenweb\core\helpers\UrlHelper;
use Respect\Validation\Validator as v;

abstract class Whmcs extends App
{
    public $moduleName;

    public $whmcsRoot;

    public $templateDir = 'templates';

    public $clientTemplateDir = 'client';

    public $forceCompileTemplates = false;

    public $showDebugTabFor = [];

    public $language = 'farsi';

    public $denyAllExcept = ['*'];

    /**
     * @var Setting
     */
    public $setting;

    public $errorsLevel = E_ERROR | E_STRICT;

    /**
     * @var \Smarty
     */
    public $adminSmarty;

    public function beforeInit()
    {
        parent::beforeInit();
        $this->setting = static::createObject($this->setting);
        $this->adminSmarty = new \Smarty();

        if (empty($this->moduleName)) {
            throw new \Exception('The "moduleName" can not be null.');
        }

        if (empty($this->whmcsRoot)) {
            throw new \Exception('The "whmcsRoot" can not be null.');
        }

        if (empty($this->templateDir)) {
            throw new \Exception('The "templateDir" can not be null.');
        }

        $this->templateDir = FileHelper::standard($this->templateDir, true);
        $this->clientTemplateDir = FileHelper::standard($this->clientTemplateDir, true);
        $this->denyDirectAccessToModuleFiles();
    }

    public function getTemplateDir()
    {
        return static::rootDir() . $this->templateDir;
    }

    public function getClientTemplateDir()
    {
        return static::rootDir() . $this->clientTemplateDir;
    }

    public function getLanguages()
    {
        $ds = DIRECTORY_SEPARATOR;
        $file = __DIR__ . $ds . "lang" . $ds . $this->language . ".php";
        return require $file;
    }

    public function routeAdminArea($_page, $_vars)
    {
        $ds = DIRECTORY_SEPARATOR;
        $_dir = $this->getTemplateDir();
        $_baseDir = __DIR__ . $ds . "baseActions";
        $_link = $_vars['modulelink'];

        $smarty = $this->adminSmarty;
        $smarty->error_reporting = E_ERROR;
        $smarty->force_compile = $this->forceCompileTemplates;
        $smarty->assign('page', $_page);
        $smarty->assign('_LANG', array_merge($this->getLanguages(), $_vars['_lang']));
        $smarty->assign('WEB_ROOT', UrlHelper::standard($this->setting->configuration()['SystemURL']));
        $smarty->assign('req', $this->requirements());
        $smarty->assign('showDebugTab', in_array(@$this->session['adminid'], $this->showDebugTabFor));
        $smarty->assign('templateDir', $_dir);
        $smarty->assign('baseTemplateDir', $_baseDir);
        $smarty->assign('modulelink', $_link);
        $smarty->registerPlugin('modifier', 'in_array', function ($needle, $haystack, $strict = false) {
            return in_array($needle, is_array($haystack) ? $haystack : [], $strict);
        });

        if (v::alnum()->validate($_page)) {
            if (($selectedDir = FileHelper::findFileInPaths("$_page.php", [$_dir, $_baseDir])) !== false) {
                require $selectedDir . $ds . "$_page.php";
                $smarty->display($selectedDir . $ds . "$_page.tpl");
                return true;
            }
        }
        return false;
    }

    public function routeClientArea($page, $_vars)
    {
        $ds = DIRECTORY_SEPARATOR;
        $dir = $this->getClientTemplateDir();

        if (file_exists($dir . $ds . "$page.php")) {
            $lang = array_merge($this->getLanguages(), $_vars['_lang']);

            include_once $dir . $ds . "$page.php";
            return [
                'pagetitle' => $lang['title'],
                'templatefile' => $this->clientTemplateDir . $ds . $page,
                'requirelogin' => true,
                'vars' => [
                    '_LANG' => $lang,
                    'WEB_ROOT' => UrlHelper::standard($this->setting->configuration()['SystemURL'])
                ]
            ];
        }

        return false;
    }

    public function denyDirectAccessToModuleFiles()
    {
        $file = static::rootDir() . "/.htaccess";
        $key = "ht-access-key";
        $cache = new ArrayCache(['app' => $this, 'array' => $this->denyAllExcept]);
        if ($cache->get($key)) {
            return true;
        }

        if (in_array('*', $this->denyAllExcept)) {
            $cache->set($key, $key);
            return FileHelper::delete($file);
        }

        $allow = "";
        foreach ($this->denyAllExcept as $except) {
            $allow .= preg_quote($except) . "|";
        }
        $allow = rtrim($allow, "|");

        $cache->set($key, $key);
        return file_put_contents($file,

            "################################################" . PHP_EOL .
            "# This file generated automatically by module. #" . PHP_EOL .
            "# please do not edit this file or your changes #" . PHP_EOL .
            "# will erase immediately after next request.   #" . PHP_EOL .
            "# thanks in advance. (Pooya)                   #" . PHP_EOL .
            "################################################" . PHP_EOL .
            "RewriteEngine on"                                 . PHP_EOL .
            "RewriteRule !^($allow)/.*$ - [F]"

        );
    }

    public function prototyper()
    {
        (new Prototyper())->run();
    }
}