<?php
namespace greenweb\core\whmcs;

use greenweb\core\helpers\FileHelper;
use greenweb\core\models\Model;
use WHMCS\Config\Application;
use Respect\Validation\Validator as v;
use Illuminate\Database\Capsule\Manager as Capsule;

/**
 * Class Ticket
 * @package greenweb\core\whmcs
 *
 * @property Whmcs $app
 *
 * @property int $deptid The department to open the ticket in. Required
 * @property string $subject The subject of the ticket. Required
 * @property string $message The message of the ticket. Required
 * @property int $clientid If applicable, the Client ID to create the ticket for. Optional
 * @property int $contactid If applicable, the Contact ID to create the ticket for (only if $clientid is passed). Optional
 * @property string $name The name of the person opening the ticket (if not a client). Optional
 * @property string $email The email address of the person opening the ticket (if not a client). Optional
 * @property string $priority The priority of the ticket (‘Low’, ‘Medium’, ‘High’). Optional
 * @property int $serviceid The service to associate the ticket with (only one of $serviceid or $domainid). Optional
 * @property int $domainid The domain to associate the ticket with (only one of $serviceid or $domainid). Optional
 * @property bool $admin Is an Admin opening the ticket. Optional
 * @property bool $useMarkdown Should markdown be used on the ticket output. Optional
 * @property string $customfields Base64 encoded serialized array of custom field values. Optional
 * @property string $attachment Separate via pipe. Optional
 */
class Ticket extends Model
{
    const TABLE_NAME = 'tbltickets';

    /**
     * @var string Path of upload dir.
     */
    private $uploadPath;

    /**
     * @var string Dir of whmcs.
     */
    public $whmcsDir;

    /**
     * @var string Username of admin.
     */
    public $adminUser;

    public $deptid;
    public $subject;
    public $message;
    public $clientid;
    public $contactid;
    public $name;
    public $email;
    public $priority = 'Medium';
    public $serviceid;
    public $domainid;
    public $admin = true;
    public $useMarkdown = false;
    public $customfields;
    public $attachment;

    public function init()
    {
        if ($this->whmcsDir === null) {
            throw new \Exception('"whmcsDir" can not be null.');
        }

        if ($this->adminUser === null) {
            throw new \Exception('"adminUser" can not be null.');
        }

        $config = new Application([]);
        $config->loadConfigFile($this->whmcsDir . "/configuration.php");
        $this->uploadPath = $config->getAbsoluteAttachmentsPath();
    }

    public function rules()
    {
        return [
            'subject' => v::notEmpty()->validate($this->subject),
            'message' => v::notEmpty()->validate($this->message)
        ];
    }

    public function save($validate = true)
    {
        if ($validate and !$this->validate()) {
            return false;
        }

        $command = "openticket";
        $values = [];
        foreach ($this->attributes() as $field) {
            if (($val = $this->{$field}) !== null) {
                $values[$field] = $val;
            }
        }
        $apiResult = localAPI($command, $values, $this->adminUser);

        if ($apiResult['result'] == 'success') {
            Capsule::table(self::TABLE_NAME)->where('id', '=', $apiResult['id'])
                ->update(['attachment' => $this->attachment]);
        }

        return $apiResult;
    }

    public function attachFiles($name)
    {
        $files = $_FILES[$name];
        foreach ($files['name'] as $i => $name) {
            $error = $files['error'][$i];
            $type = $files['type'][$i];
            $tmp = $files['tmp_name'][$i];
            if ($error === UPLOAD_ERR_OK) {
                $res = mt_rand(100000, 999999) ."_$name";
                if (copy($tmp, FileHelper::standard($this->uploadPath) . "/$res")) {
                    $this->attachment .= $res.'|';
                }
            }
        }
        $this->attachment = rtrim($this->attachment, '|');
    }

    public function attributes()
    {
        return [
            'deptid', 'subject', 'message', 'clientid', 'contactid', 'name',
            'email', 'priority', 'serviceid', 'domainid', 'admin', 'useMarkdown', 'customfields'
        ];
    }
}