<?php
namespace greenweb\core\helpers;

class FileHelper
{
    /**
     * @param $path string Input directory path which you want to make standard.
     * @param $preSlash bool add slash before path.
     * @return string Standard path without end slash.
     */
    public static function standard($path, $preSlash = false)
    {
        $path = rtrim(str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path), DIRECTORY_SEPARATOR);
        return $preSlash ? DIRECTORY_SEPARATOR . ltrim($path, DIRECTORY_SEPARATOR) : $path;
    }

    /**
     * @param $dir string Input dir.
     * @return array List of all files in selected directory with flat style.
     */
    public static function scanDirRecursive($dir)
    {
        $dir = self::standard($dir);
        $return = [];
        $files = scandir($dir);
        if($files) foreach($files as $item) {
            if($item == '.' or $item == '..') {
                continue;
            }
            $fullName = $dir . DIRECTORY_SEPARATOR . $item;
            if(is_file($fullName)) {
                $return[] = $fullName;
            } else {
                $return = array_merge($return, self::scanDirRecursive($fullName));
            }
        }
        return $return;
    }

    const COPY_INNER_FILES = 1;

    /**
     * @param $src string Path to source folder or file.
     * @param $destDir string Path to destination folder.
     * @param array $options List of custom behavior options.
     *  Modes:
     *   - COPY_INNER_FILES: when you want to copy inner files of a folder. not selected folder.
     *
     * @throws \Exception When source is not a valid file or folder or destination is not a valid dir.
     */
    public static function copy($src, $destDir, array $options = [])
    {
        $destDir = self::standard($destDir);

        if (!is_dir($destDir)) {
            throw new \Exception('The "destDir" is not a valid directory.');
        }

        if (is_dir($src)) {
            $src = self::standard($src);
            if (!in_array(self::COPY_INNER_FILES, $options)) {
                $destDir = $destDir . DIRECTORY_SEPARATOR . basename($src);
                if (!is_dir($destDir)) {
                    mkdir($destDir);
                }
            }

            if ($files = scandir($src)) {
                foreach ($files as $item) {
                    if($item != '.' and $item != '..') {
                        self::copy($src . DIRECTORY_SEPARATOR . $item, $destDir);
                    }
                }
            }
        }
        elseif (is_file($src)) {
            copy($src, $destDir . DIRECTORY_SEPARATOR . basename($src));
        }
        else {
            throw new \Exception('The "src" is not a valid filename or directory.');
        }
    }

    /**
     * @param string $filename Which you want to find it in paths.
     * @param array $paths List of paths.
     * @return bool|string Path name if file exists in that. otherwise return false.
     */
    public static function findFileInPaths($filename, array $paths)
    {
        foreach ($paths as $path) {
            $path = self::standard($path);
            if (is_file($path . DIRECTORY_SEPARATOR . $filename)) {
                return $path;
            }
        }
        return false;
    }

    /**
     * @param string $path Which you want to delete.
     * @return bool @see unlink().
     */
    public static function delete($path)
    {
        try {
            if (is_file($path)) {
                return unlink($path);
            } elseif (is_dir($path)) {
                foreach (scandir($path) as $file) {
                    if ($file == '.' or $file == '..') {
                        continue;
                    }
                    $file = $path . DIRECTORY_SEPARATOR . $file;
                    self::delete($file);
                }
                return rmdir($path);
            } else {
                return false;
            }
        } catch (\Exception $ex) {
            return false;
        }
    }
}